#!/usr/bin/env python3.7
"""Next Gen Common code, even for downloader and other tools"""
#
# -*-mode: python; coding: utf-8 -*-
#
#
# Copyright 2020-2021 Garo AB

import os
import sys
import atexit

UNIT_TYPE_CHARGING_UNIT = "GaroCU"
UNIT_TYPE_LOADINTERFACE = "GaroLI"

NG_VERSION = None

def get_version():
    """Get software version"""
    try:
        bbfile = os.path.join(os.path.dirname(__file__), "../../../garoevse.bb")
        with open(bbfile) as obb:
            for line in obb:
                line = line.strip()
                if line.startswith("PV "):
                    ver = line.split(None, 2)[2]
                    ver = ver.replace('"', '')
                    return ver
    except FileNotFoundError:
        pass

    return NG_VERSION

def serial_to_mac(serial, iface):
    "Convert a serial and interface number to MAC address, or None"
    mac = serial >> 4
    if iface > 15:
        return None
    mac = (mac << 4) + iface
    if mac == serial:
        # If resulting MAC is same as serial, this means that the
        # serial (first Ethernet MAC) did not end with zero but
        # instead a value which collides with specified interface
        return None

    return mac

def mac_to_readable(mac):
    "Print number as readable MAC address"
    # Doing manually to avoid netaddr dependency
    return "%.02x:%.02x:%.02x:%.02x:%.02x:%.02x" % tuple(mac.to_bytes(6, "big"))

def get_unit_serial(config=None):
    "Return the serial number of this charging unit, or zero upon failures"
    if config is not None:
        unit_serial = config["ocpp16"].get("GaroUnitSerial", "")
        if unit_serial:
            return int(unit_serial, 16)

    # The serial number is MAC of the first Ethernet. On the logic
    # board, we should thus look at eth0 or eth1, but in order to
    # support other systems, fallback to any interface. Prefer eth1,
    # since this corresponds to ENET1 on imx6 (uboot ethaddr environment
    # variable).
    ifaces = ["eth1", "eth0", "wlan0"]
    try:
        ifaces += sorted(os.listdir("/sys/class/net"))
    except FileNotFoundError:
        pass
    addr = 0
    for iface in ifaces:
        try:
            with open(f"/sys/class/net/{iface}/address", "r") as sysaddr:
                addr = sysaddr.read().strip()
                addr = addr.replace(":", "")
                addr = int(addr, base=16)
                if addr != 0:
                    break
        except FileNotFoundError:
            pass

    # A MAC address has 48 bits and is typically formatted as: OP:QR:ST:UV:WX:YZ
    #
    # These bits has been allocated as:
    # OP:QR:ST          Organizationally Unique Identifier (Garo Next Gen prefix)
    # UV:WX:Y           Unique Number (max 0xfffff)
    # Z                 Network Interface Identifier (max 0xf)
    #
    # A serial number normally ends with 0, since it is defined as MAC
    # of first Ethernet interface.
    #
    # For the Network Interface Identifier, these are allocated as:
    # 0-3 Ethernet
    # 4-7 Wifi
    # 8 PLC/15118
    return addr

def get_device_id(config=None, bracket_id=None):
    "Get the installation bracket ID, either manually configured, or use Bracket ID"
    if config is not None:
        device_id = config["ocpp16"].get("GaroDeviceId", "")
        if device_id:
            return device_id

    unit_serial = get_unit_serial(config)
    if get_unit_type(config) == UNIT_TYPE_LOADINTERFACE:
        return f"GaroLI-{unit_serial:X}"

    if bracket_id is not None:
        bracket_id = bracket_id.upper().lstrip("0")
        return f"GaroCS-{bracket_id}"

    # If RFID communication fails, fall back to using "unit"
    # ID. Connecting to CSMS with the "wrong" ID is better than no
    # connection at all. This fallback is also used on BeagleBone.
    return f"GaroCU-{unit_serial:X}"


def get_unit_type(config=None):
    "Return UnitType.CHARGING_UNIT or UnitType.LOADINTERFACE depending on type of unit"
    if "Load Interface" in get_device_model(config):
        return UNIT_TYPE_LOADINTERFACE
    return UNIT_TYPE_CHARGING_UNIT


def get_device_model(config=None):
    """Get the device model type using the device-tree model name, or
    GaroDeviceModel parameter. The string contains "Load Interface"
    and "BeagleBone" on those platforms.
    """
    if config is not None:
        # If GaroDeviceModel is set, use it.
        device_model = config["ocpp16"].get("GaroDeviceModel", "")
        if device_model:
            return device_model
    try:
        with open("/proc/device-tree/model") as dtmodel:
            return dtmodel.read()
    except FileNotFoundError:
        pass

    return "Unknown"


def restart_application():
    "Restart the software application"
    sys.stdout.flush()
    sys.stderr.flush()
    try:
        atexit._run_exitfuncs()  # pylint:disable=protected-access
    except Exception as ex: # pylint: disable=broad-except
        print(f"Exception during atexit: {ex}", file=sys.stderr)
    os.execv(sys.executable, [sys.executable] + sys.argv)
