#!/usr/bin/env python3.7
"""Entity Common code, even for downloader and other tools"""
#
# -*-mode: python; coding: utf-8 -*-
#
#
# Copyright 2020-2021 Garo AB

__all__ = ["NG_VERSION", "NG_REVISION",
           "remove_stop_transaction_on_invalid_id_file",
           "get_connector_locked_invalid_transaction_id",
           "set_connector_locked_invalid_transaction_id",
           "get_version", "get_revision",
           "serial_to_mac", "mac_to_readable",
           "get_unit_serial", "get_device_id",
           "get_unit_type", "get_device_model",
           "add_device_id_and_check_url", "restart_application",
           "get_bracket_phase_rotation", "open_utf8",
           "UNIT_TYPE_CHARGING_UNIT", "UNIT_TYPE_LOADINTERFACE",
           "METERVALUES_SHORTEN_DICT", "EM_READINGS_SHORTEN_DICT"]

import os
import sys
import atexit
import subprocess
import logging
import urllib.parse

UNIT_TYPE_CHARGING_UNIT = "GaroCU"
UNIT_TYPE_LOADINTERFACE = "GaroLI"

NG_VERSION = None
NG_REVISION = None

# Translation table for shortening meter values.
METERVALUES_SHORTEN_DICT = {"'measurand'": "'m'",
                            "'timestamp'": "'t'",
                            "'sampledValue'": "'s'",
                            "'context'": "'c'",
                            "'Sample.Clock'": "'S.C'",
                            "'Trigger'": "'Tr'",
                            "'location'": "'l'",
                            "'value'": "'v'",
                            "'unit'": "'u'",
                            "'phase'": "'p'",
                            "'Voltage'": "'Vo'",
                            "'Temperature'": "'T'",
                            "'Celsius'": "'C'",
                            "'Outlet'": "'O'",
                            "'Inlet'": "'I'",
                            "'Body'": "'B'"
}

# Translation table for shortening energy meter readings.
EM_READINGS_SHORTEN_DICT = {
     "Energy.Active.Import.Register": "EAIR",
     "Energy.Active.Export.Register": "EAER",
     "Energy.Reactive.Import.Register": "ERIR",
     "Energy.Reactive.Export.Register": "ERER",
     "Power.Active.Import": "PAI",
     "Power.Active.Export": "PAE",
     "Power.Reactive.Import": "PRI",
     "Power.Reactive.Export": "PRE",
     "Voltage.L1-N": "VL1-N",
     "Voltage.L2-N": "VL2-N",
     "Voltage.L3-N": "VL3-N",
     "Voltage.L1-L2": "VL1-L2",
     "Voltage.L2-L3": "VL2-L3",
     "Voltage.L3-L1": "VL3-L1",
     "Current.Import": "CI",
     "Current.Export": "CE",
     "Current.Offered": "CO",
     "Serial.Number": "SN"
}

FILE_PATH = file_path = os.path.join(os.path.dirname(__file__), "../../stop_transaction_on_invalid_id")

def remove_stop_transaction_on_invalid_id_file():
    """
    Remove file keeping track of locked connector because of invalid id.
    """
    try:
        os.remove(FILE_PATH)
    except FileNotFoundError:
        pass

def get_connector_locked_invalid_transaction_id():
    """
    Open file 'stop_transaction_on_invalid_id' to check if connector is currently locked,
    return tag that was responsible for locking connector. If connector not locked, return None.
    """
    tag = None
    parent_tag = None
    try:
        # Check if connector is locked because of a invalid transaction occured.
        with open_utf8(FILE_PATH, "r") as file:
            # First line tag, second line parent id tag (if used, may be empty).
            tag = file.readline()
            parent_tag = file.readline()
    except FileNotFoundError:
        pass
    if tag is not None:
        tag = tag.strip()
    if parent_tag == "":
        parent_tag = None
    # None returned means connector not locked,
    # else if tag is set from file it means connector is locked and by that tag (the tag can also have a parent tag).
    return tag, parent_tag

def set_connector_locked_invalid_transaction_id(id_tag, parent_tag=""):
    """
    Open file 'stop_transaction_on_invalid_id' and write id_tag that was responsible for locking connector.
    """
    with open_utf8(FILE_PATH, "w") as file:
        if parent_tag is not None:
            file.writelines([id_tag + '\n', parent_tag])
        else:
            file.write(id_tag + '\n')

def get_version():
    """Get software version"""
    if NG_VERSION is not None:
        return NG_VERSION

    try:
        bbfile = os.path.join(os.path.dirname(__file__), "../../../garoevse.bb")
        with open_utf8(bbfile) as obb:
            for line in obb:
                line = line.strip()
                if line.startswith("PV "):
                    ver = line.split(None, 2)[2]
                    ver = ver.replace('"', '')
                    return ver
    except FileNotFoundError:
        pass

    return None

def get_revision():
    """Get software revision"""
    if NG_REVISION is not None:
        return NG_REVISION

    try:
        git = subprocess.run(["git", "rev-parse", "--short", "HEAD"], stdout=subprocess.PIPE, text=True)
        return git.stdout.strip()
    except OSError:
        pass

    return None

def serial_to_mac(serial, iface):
    "Convert a serial and interface number to MAC address, or None"
    mac = serial >> 4
    if iface > 15:
        return None
    mac = (mac << 4) + iface
    if mac == serial:
        # If resulting MAC is same as serial, this means that the
        # serial (first Ethernet MAC) did not end with zero but
        # instead a value which collides with specified interface
        return None

    return mac

def mac_to_readable(mac):
    "Print number as readable MAC address"
    # Doing manually to avoid netaddr dependency
    return "%.02x:%.02x:%.02x:%.02x:%.02x:%.02x" % tuple(mac.to_bytes(6, "big"))

def get_unit_serial(config=None):
    "Return the serial number of this charging unit, or zero upon failures"
    if config is not None:
        unit_serial = config["ocpp16"].get("GaroUnitSerial")
        if unit_serial:
            return int(unit_serial, 16)

    # The serial number is MAC of the first Ethernet. On the logic
    # board, we should thus look at eth0 or eth1, but in order to
    # support other systems, fallback to any interface. Prefer eth1,
    # since this corresponds to ENET1 on imx6 (uboot ethaddr environment
    # variable).
    ifaces = ["eth1", "eth0", "wlan0"]
    try:
        ifaces += sorted(os.listdir("/sys/class/net"))
    except FileNotFoundError:
        pass
    addr = 0
    for iface in ifaces:
        try:
            with open_utf8(f"/sys/class/net/{iface}/address", "r") as sysaddr:
                addr = sysaddr.read().strip()
                addr = addr.replace(":", "")
                addr = int(addr, base=16)
                if addr != 0:
                    break
        except FileNotFoundError:
            pass

    # A MAC address has 48 bits and is typically formatted as: OP:QR:ST:UV:WX:YZ
    #
    # These bits has been allocated as:
    # OP:QR:ST          Organizationally Unique Identifier (Garo Entity prefix)
    # UV:WX:Y           Unique Number (max 0xfffff)
    # Z                 Network Interface Identifier (max 0xf)
    #
    # A serial number normally ends with 0, since it is defined as MAC
    # of first Ethernet interface.
    #
    # For the Network Interface Identifier, these are allocated as:
    # 0-3 Ethernet
    # 4-7 Wi-Fi
    # 8 PLC/15118
    return addr

def get_device_id(config=None, bracket_id=None):
    "Get the installation bracket ID, either manually configured, or use Bracket ID"
    if config is not None:
        device_id = config["ocpp16"].get("GaroDeviceId")
        if device_id:
            return device_id

    unit_serial = get_unit_serial(config)
    if get_unit_type(config) == UNIT_TYPE_LOADINTERFACE:
        return f"GaroLI-{unit_serial:X}"

    if bracket_id is not None:
        bracket_id = bracket_id.upper().lstrip("0")
        return f"GaroCS-{bracket_id}"

    # If RFID communication fails, fall back to using "unit"
    # ID. Connecting to CSMS with the "wrong" ID is better than no
    # connection at all.
    return f"GaroCU-{unit_serial:X}"


def get_unit_type(config=None):
    "Return UnitType.CHARGING_UNIT or UnitType.LOADINTERFACE depending on type of unit"
    if "Load Interface" in get_device_model(config):
        return UNIT_TYPE_LOADINTERFACE
    return UNIT_TYPE_CHARGING_UNIT


def get_device_model(config=None):
    """Get the device model type using the device-tree model name, or
    GaroDeviceModel parameter. The string contains "Load Interface"
    and "Compact" on those platforms.
    """
    if config is not None:
        # If GaroDeviceModel is set, use it.
        device_model = config["ocpp16"].get("GaroDeviceModel", fallback="")
        if device_model:
            return device_model
    try:
        with open_utf8("/proc/device-tree/model") as dtmodel:
            return dtmodel.read()
    except FileNotFoundError:
        pass

    return "Unknown"

def add_device_id_and_check_url(endpoint_url, device_id, profile):
    """Performs a basic url check towards SecurityProfile setting and also
    adds schema in case it was not specified.
    """
    # Check SecurityProfile
    if profile not in (0, 1, 2, 3):
        logging.warning(f"Invalid security profile={profile}, falling back to 2")
        profile = 2

    # Add schema if it is undefined in endpoint url
    if len(endpoint_url.split('://')) == 1:
        if profile == 1:
            endpoint_url = "ws://" + endpoint_url
        elif profile == 2:
            endpoint_url = "wss://" + endpoint_url

    # Remove trailing slash from URL to avoid extra slashes when adding device_id
    url = list(urllib.parse.urlsplit(endpoint_url.rstrip('/')))
    # Append identity
    url[2] = url[2] + "/" + device_id

    # Determine and setup security profile. Empty password only
    # supported with security profile 0.
    if profile in (0, 1) and url[0] != "ws":
        logging.warning(f"Security profile={profile} does not match URL, using URL scheme {url[0]} instead of 'ws'")
    if profile in (2, 3) and url[0] != "wss":
        logging.warning(f"Security profile={profile} does not match URL, using URL scheme {url[0]} instead of 'wss'")
    if not endpoint_url.startswith(url[0] + ":"):
        logging.warning(f"Invalid CSMS URL scheme with profile={profile}, falling back to {url[0]}")

    # Also return profile, since it might have been altered by above checks
    return urllib.parse.urlunsplit(url), profile

def restart_application():
    "Restart the software application"
    sys.stdout.flush()
    sys.stderr.flush()
    # Change back fd 1/2 from std_socket to real stdout/stderr
    os.dup2(sys.stdout.fileno(), 1)
    os.dup2(sys.stderr.fileno(), 2)
    try:
        atexit._run_exitfuncs()  # pylint:disable=protected-access
    except Exception as ex: # pylint: disable=broad-except
        print(f"Exception during atexit: {ex}", file=sys.stderr)
    os.execv(sys.executable, [sys.executable] + sys.argv)


def get_bracket_phase_rotation(config):
    "Get ConnectorPhaseRotation for bracket as list of integers"
    rotation = "RST"
    cpr = config.get_conf_csl_element("ConnectorPhaseRotation", 0)
    cpr = cpr.upper()
    try:
        # Make sure we only have R,S,T letters and correct length
        if cpr.translate(cpr.maketrans("", "", "RST")) != "":
            raise ValueError
        if len(cpr) != 3:
            raise ValueError
        rotation = cpr
    except ValueError:
        logging.warning(f"Invalid ConnectorPhaseRotation[0]={cpr}")
    # Translate string with RST format to list of indices, and rotate
    rotation = rotation.translate(rotation.maketrans("RST", "012"))
    # Translate this to list like [1, 2, 0]
    rotation = list(map(int, rotation))
    return rotation

def open_utf8(*args, **kwargs):
    "Common function för opening files as UTF-8"
    return open(*args, encoding="utf-8", errors="surrogateescape", **kwargs)
